# --
# Copyright (C) 2001-2019 OTRS AG, https://otrs.com/
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see https://www.gnu.org/licenses/gpl-3.0.txt.
# --

use strict;
use warnings;
use utf8;

use vars (qw($Self));

# get selenium object
my $Selenium = $Kernel::OM->Get('Kernel::System::UnitTest::Selenium');

$Selenium->RunTest(
    sub {

        # get helper object
        my $Helper = $Kernel::OM->Get('Kernel::System::UnitTest::Helper');

        # do not check RichText
        $Helper->ConfigSettingChange(
            Valid => 1,
            Key   => 'Frontend::RichText',
            Value => 0
        );

        # Defined user language for testing if message is being translated correctly.
        my $Language = "de";

        # create test user and login
        my $TestUserLogin = $Helper->TestUserCreate(
            Groups   => ['admin'],
            Language => $Language,
        ) || die "Did not get test user";

        $Selenium->Login(
            Type     => 'Agent',
            User     => $TestUserLogin,
            Password => $TestUserLogin,
        );

        # get script alias
        my $ScriptAlias = $Kernel::OM->Get('Kernel::Config')->Get('ScriptAlias');

        # navigate to AdminAutoResponse screen
        $Selenium->VerifiedGet("${ScriptAlias}index.pl?Action=AdminAutoResponse");

        # check overview AdminAutoResponse
        $Selenium->find_element( "table",             'css' );
        $Selenium->find_element( "table thead tr th", 'css' );
        $Selenium->find_element( "table tbody tr td", 'css' );

        # click 'Add auto response'
        $Selenium->find_element("//a[contains(\@href, \'Action=AdminAutoResponse;Subaction=Add' )]")->VerifiedClick();

        # check page
        for my $ID (
            qw(Name Subject RichText TypeID AddressID ValidID Comment)
            )
        {
            my $Element = $Selenium->find_element( "#$ID", 'css' );
            $Element->is_enabled();
            $Element->is_displayed();
        }

        # check client side validation
        my $Element = $Selenium->find_element( "#Name", 'css' );
        $Element->send_keys("");
        $Selenium->find_element("//button[\@type='submit']")->VerifiedClick();

        $Self->Is(
            $Selenium->execute_script(
                "return \$('#Name').hasClass('Error')"
            ),
            '1',
            'Client side validation correctly detected missing input value',
        );

        # create a real test Auto Response
        my $RandomID = 'AutoResponse' . $Helper->GetRandomID();
        my $Text     = "Selenium auto response text";

        $Selenium->find_element( "#Name",     'css' )->send_keys($RandomID);
        $Selenium->find_element( "#Subject",  'css' )->send_keys($RandomID);
        $Selenium->find_element( "#RichText", 'css' )->send_keys($Text);
        $Selenium->execute_script("\$('#TypeID').val('1').trigger('redraw.InputField').trigger('change');");
        $Selenium->execute_script("\$('#AddressID').val('1').trigger('redraw.InputField').trigger('change');");
        $Selenium->execute_script("\$('#ValidID').val('1').trigger('redraw.InputField').trigger('change');");
        $Selenium->find_element("//button[\@type='submit']")->VerifiedClick();

        # check if test auto response show on AdminAutoResponse screen
        $Self->True(
            index( $Selenium->get_page_source(), $RandomID ) > -1,
            "$RandomID job found on page",
        );

        # edit test job and set it to invalid
        $Selenium->find_element( $RandomID, 'link_text' )->VerifiedClick();
        my $RandomID2 = 'AutoResponseUpdate' . $Helper->GetRandomID();
        $Selenium->find_element( "#Name", 'css' )->clear();
        $Selenium->find_element( "#Name", 'css' )->send_keys($RandomID2);
        $Selenium->execute_script("\$('#ValidID').val('2').trigger('redraw.InputField').trigger('change');");
        $Selenium->find_element("//button[\@type='submit']")->VerifiedClick();

        # check if edited auto response show on AdminAutoResponse
        $Self->True(
            index( $Selenium->get_page_source(), $RandomID2 ) > -1,
            "$RandomID2 auto response found on page",
        );

        # check class of invalid AutoResponse in the overview table
        $Self->True(
            $Selenium->execute_script(
                "return \$('tr.Invalid td a:contains($RandomID2)').length"
            ),
            "There is a class 'Invalid' for test AutoResponse",
        );

        # Navigate to AdminAutoResponse screen.
        $Selenium->VerifiedGet("${ScriptAlias}index.pl?Action=AdminAutoResponse");

        my $LanguageObject = Kernel::Language->new(
            UserLanguage => $Language,
        );

        my $Count = 0;
        for my $ColumnName (qw(Name Type Comment Validity Changed Created)) {

            # Check if column name is translated.
            $Self->Is(
                $Selenium->execute_script("return \$('.DataTable tr th:eq($Count)').text().trim()"),
                $LanguageObject->Translate($ColumnName),
                "Column name $ColumnName is translated",
            );
            $Count++;
        }

        # since there are no tickets that rely on our test auto response, we can remove them
        # again from the DB
        if ($RandomID2) {
            my $DBObject = $Kernel::OM->Get('Kernel::System::DB');
            $RandomID2 = $DBObject->Quote($RandomID2);
            my $Success = $DBObject->Do(
                SQL  => "DELETE FROM auto_response WHERE name = ?",
                Bind => [ \$RandomID2 ],
            );
            if ($Success) {
                $Self->True(
                    $Success,
                    "AutoResponseDelete - $RandomID2",
                );
            }
        }

    }

);

1;
